<?php
/**
* MercadoPago Integration Library, Access MercadoPago for payments integration.
* 
* @author    hcasatti
* @copyright 2007-2011 MercadoPago
* @license   http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
*/

class MercadoPagoGenericMpmxs {

    const version = "0.2";

    private $client_id;
    private $client_secret;
    private $access_data = null;
    private $sandbox = FALSE;
    private $file_name = FALSE;

    function __construct($client_id, $client_secret) {
        $this->client_id = $client_id;
        $this->client_secret = $client_secret;
        $this->file_name = dirname(__FILE__).'/.'.$this->client_id.'-token.php';
    }

    public function sandbox_mode($enable = NULL) {
        if (!is_null($enable)) {
            $this->sandbox = $enable === TRUE;
        }

        return $this->sandbox;
    }

    /**
     * Get Access Token for API use
     */
    public function get_access_token() {
        if (is_null($this->access_data)) {
            $data = @json_decode(@trim(@str_replace('<?php exit; ?>', '', @file_get_contents($this->file_name))), true);
            if ($data && isset($data['expire'])) {
                if ($data['expire'] > time()) {
                    $this->access_data = $data['data'];
                    return $this->access_data['access_token'];
                }
            }
            $appClientValues = $this->build_query(array(
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
                'grant_type' => 'client_credentials'
            ));
            $access_data = MPRestClientGenericMpmxs::post("/oauth/token", $appClientValues, "application/x-www-form-urlencoded");
            $this->access_data = $access_data['response'];
            if (isset($this->access_data['expires_in']) && isset($this->access_data['access_token'])) {
                $data = array('expire' => time() + $this->access_data['expires_in'] - 10, 'data' => $this->access_data);
                @unlink($this->file_name);
                @file_put_contents($this->file_name, '<?php exit; ?>'.json_encode($data));
            }
        }

        return is_array($this->access_data) && isset($this->access_data['access_token'])?$this->access_data['access_token']:false;
    }

    /**
     * Get information for specific payment
     * @param int $id
     * @return array(json)
     */
    public function get_payment_info($id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $uriPrefix = $this->sandbox ? "/sandbox" : "";
            
        $response = MPRestClientGenericMpmxs::get($uriPrefix."/collections/notifications/" . $id . "?access_token=" . $accessToken);
        
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_payment_info($id, false);
        }
        
        return $response;
    }
    public function get_merchant_order($id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $uriPrefix = $this->sandbox ? "/sandbox" : "";

        $response = MPRestClientGenericMpmxs::get($uriPrefix."/merchant_orders/" . $id . "?access_token=" . $accessToken);
        
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_merchant_order($id, false);
        }
        
        return $response;
    }

    /**
     * Refund accredited payment
     * @param int $id
     * @return array(json)
     */
    public function refund_payment($id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $refund_status = array(
            "status" => "refunded"
        );

        $response = MPRestClientGenericMpmxs::put("/collections/" . $id . "?access_token=" . $accessToken, $refund_status);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->refund_payment($id, false);
        }
        
        return $response;
    }

    /**
     * Cancel pending payment
     * @param int $id
     * @return array(json)
     */
    public function cancel_payment($id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $cancel_status = array(
            "status" => "cancelled"
        );

        $response = MPRestClientGenericMpmxs::put("/collections/" . $id . "?access_token=" . $accessToken, $cancel_status);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->cancel_payment($id, false);
        }
        
        return $response;
    }

    /**
     * Search payments according to filters, with pagination
     * @param array $filters
     * @param int $offset
     * @param int $limit
     * @return array(json)
     */
    public function search_payment($filters, $offset = 0, $limit = 0, $first_call = true) {
        $accessToken = $this->get_access_token();

        $filters["offset"] = $offset;
        $filters["limit"] = $limit;

        $filters = $this->build_query($filters);

        $uriPrefix = $this->sandbox ? "/sandbox" : "";
            
        $response = MPRestClientGenericMpmxs::get($uriPrefix."/collections/search?" . $filters . "&access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->search_payment($filters, $offset, $limit, false);
        }
        
        return $response;
    }

    /**
     * Create a checkout preference
     * @param array $preference
     * @return array(json)
     */
    public function create_preference($preference, $first_call = true) {
        $accessToken = $this->get_access_token();

        $response = MPRestClientGenericMpmxs::post("/checkout/preferences?access_token=" . $accessToken, $preference);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->create_preference($preference, false);
        }
        
        return $response;
    }

    /**
     * Update a checkout preference
     * @param string $id
     * @param array $preference
     * @return array(json)
     */
    public function update_preference($id, $preference, $first_call = true) {
        $accessToken = $this->get_access_token();

        $response = MPRestClientGenericMpmxs::put("/checkout/preferences/{$id}?access_token=" . $accessToken, $preference);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->update_preference($id, $preference, false);
        }
        
        return $response;
    }

    /**
     * Get a checkout preference
     * @param string $id
     * @return array(json)
     */
    public function get_preference($id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $response = MPRestClientGenericMpmxs::get("/checkout/preferences/{$id}?access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_preference($id, false);
        }
        
        return $response;
    }

    /**
     * Get a user info
     * @param string $id
     * @return array(json)
     */
    public function get_me($first_call = true) {
        $accessToken = $this->get_access_token();

        $response = MPRestClientGenericMpmxs::get("/users/me?access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_me(false);
        }

        return $response;
    }

    public function get_balance($user_id, $first_call = true) {
        $accessToken = $this->get_access_token();

        $response = MPRestClientGenericMpmxs::get("/users/$user_id/mercadopago_account/balance?access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_balance($user_id, false);
        }

        return $response;
    }
    
    public function get_shipping_price($params, $first_call = true) {
        $accessToken = $this->get_access_token();
        
        $data = $this->build_query($params);

        $response = MPRestClientGenericMpmxs::get("/shipping_options?{$data}&access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_shipping_price($params, false);
        }
        
        return $response;
    }
    public function get_shipping_status($id_shipping, $first_call = true) {
        $accessToken = $this->get_access_token();
        
        $response = MPRestClientGenericMpmxs::get("/shipments/{$id_shipping}?access_token=" . $accessToken);
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_shipping_status($id_shipping, false);
        }
        
        return $response;
    }
    public function get_label_shipping($id_shipping) {
        $me = $this->get_me(); //Dummy, solo para renovar token si hace falta...
        $accessToken = $this->get_access_token();
        return "https://api.mercadolibre.com/shipment_labels?shipment_ids={$id_shipping}&response_type=pdf&savePdf=Y&access_token=" . $accessToken;
    }
    public function get_test_user($site_id, $first_call = true) {
        $accessToken = $this->get_access_token();
        
        $response = MPRestClientGenericMpmxs::post("/users/test_user?access_token=" . $accessToken, '{"site_id":"' . $site_id . '"}');
        if ($first_call && $response['status'] == 401 && $response['response']['message'] == 'invalid_token') {
            @unlink($this->file_name);
            $this->access_data = null;
            return $this->get_test_user($site_id, false);
        }
        
        return $response;
    }
    public static function get_shipping_list($country) {
        $connect = curl_init("https://api.mercadolibre.com/sites/{$country}/shipping_methods");

        curl_setopt($connect, CURLOPT_USERAGENT, "MercadoPago PHP SDK v" . MercadoPagoGenericMpmxs::version);
        curl_setopt($connect, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($connect, CURLOPT_CUSTOMREQUEST, 'GET');
        curl_setopt($connect, CURLOPT_FAILONERROR, false);
        curl_setopt($connect, CURLOPT_SSL_VERIFYPEER, false);
        
        $apiResult = curl_exec($connect);
        $apiHttpCode = curl_getinfo($connect, CURLINFO_HTTP_CODE);
        
        $response = json_decode($apiResult, true);
        return $response;
    }

    /*     * **************************************************************************************** */

    private function build_query($params) {
        if (function_exists("http_build_query")) {
            return http_build_query($params);
        } else {
            foreach ($params as $name => $value) {
                $elements[] = "{$name}=" . urlencode($value);
            }

            return implode("&", $elements);
        }
    }

}

/**
 * MercadoPago cURL RestClient
 */
class MPRestClientGenericMpmxs {

    const API_BASE_URL = "https://api.mercadopago.com";

    private static function getConnect($uri, $method, $contentType) {
        $connect = curl_init(self::API_BASE_URL . $uri);

        curl_setopt($connect, CURLOPT_USERAGENT, "MercadoPago PHP SDK v" . MercadoPagoGenericMpmxs::version);
        curl_setopt($connect, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($connect, CURLOPT_CUSTOMREQUEST, $method);
		curl_setopt($connect, CURLOPT_FAILONERROR, false);  
		curl_setopt($connect, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($connect, CURLOPT_HTTPHEADER, array("Accept: application/json", "Content-Type: " . $contentType));

        return $connect;
    }

    private static function setData(&$connect, $data, $contentType) {
        if ($contentType == "application/json") {
            if (gettype($data) == "string") {
                json_decode($data, true);
            } else {
                $data = json_encode($data);
            }

            if(function_exists('json_last_error')) {
                $json_error = json_last_error();
                if ($json_error != JSON_ERROR_NONE) {
                    throw new Exception("JSON Error [{$json_error}] - Data: {$data}");
                }
            }
        }

        curl_setopt($connect, CURLOPT_POSTFIELDS, $data);
    }

    private static function exec($method, $uri, $data, $contentType) {
        $connect = self::getConnect($uri, $method, $contentType);
        if ($data) {
            self::setData($connect, $data, $contentType);
        }

        $apiResult = curl_exec($connect);
        $apiHttpCode = curl_getinfo($connect, CURLINFO_HTTP_CODE);

        $response = array(
            "status" => $apiHttpCode,
            "response" => json_decode($apiResult, true)
        );

        curl_close($connect);

        return $response;
    }

    public static function get($uri, $contentType = "application/json") {
        return self::exec("GET", $uri, null, $contentType);
    }

    public static function post($uri, $data, $contentType = "application/json") {
        return self::exec("POST", $uri, $data, $contentType);
    }

    public static function put($uri, $data, $contentType = "application/json") {
        return self::exec("PUT", $uri, $data, $contentType);
    }

}

